--[[
	Title: Plugin Helpers

	Some useful functions for ULib plugins to use for doing plugin-type things.
]]


--[[
	Table: plugins

	Holds plugin data for plugins that have registered themselves with ULib.

	Fields:

		Name - A string of the name of the plugin.
		Version - A string or number of the version of the plugin.
		Author - An optional string of the author of the plugin.
		URL - An optional string of the URL for the plugin.
]]
ULib.plugins = {} -- Any registered plugins go here


--[[
	Function: registerPlugin

	Parameters:

		pluginData - A table of plugin data in the format documented in <plugins>, above.
]]
function ULib.registerPlugin( pluginData )
	local name = pluginData.Name
	if not ULib.plugins[ name ] then
		ULib.plugins[ name ] = pluginData
	else
		table.Merge( ULib.plugins[ name ], pluginData )
		pluginData = ULib.plugins[ name ]
	end

	if SERVER then
		ULib.clientRPC( nil, "ULib.registerPlugin", pluginData )
	end
end


if SERVER then
	local function sendRegisteredPlugins( ply )
		for name, pluginData in pairs(ULib.plugins) do
			ULib.clientRPC( ply, "ULib.registerPlugin", pluginData )
		end
	end
	hook.Add( "PlayerInitialSpawn", "ULibSendRegisteredPlugins", sendRegisteredPlugins )
end

ULib.registerPlugin{
	Name          = "ULib",
	Version       = "MG5",
	Author        = "Team Ulysses",
	URL           = "http://ulyssesmod.net"
}


--[[
	Function: pluginVersionStr

	Returns a human-readable version string for plugins in a consistent format.
	The string tells users if they're using a development build (with build number/date), workshop, or release version.

	Parameters:

		name - The string of the plugin name you are querying about.

	Returns:

		A string of the version information for the specified plugin.
]]
function ULib.pluginVersionStr( name )
	local dat = ULib.plugins[ name ]
	if not dat then return nil end

	return string.format( "v%s", dat.Version )
end
